import pathlib
import logging
import os
import tempfile
import datetime
import flask_unsign.session
from datetime import timedelta
from typing import Union
from flask_unsign import sign
from flask_unsign import DEFAULT_SALT

TEMP_DATA_DIRECTORY = None
DEFAULT_SECRET_KEY = "cdc"
# Use a firefox user agent to make it a less suspicious.
USER_AGENT = "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:147.0) Gecko/20100101 Firefox/147.0"
MAX_TIME_ALLOWED = timedelta(hours=1)
_logger = logging.getLogger(__name__)

def strip_http_prefix(url: str) -> str:
    if url.startswith("http://"):
        return url[len("http://"):]
    elif url.startswith("https://"):
        return url[len("https://"):]
    return url

def check_domain_folder_exists(domain: str) -> bool:
    global TEMP_DATA_DIRECTORY
    if TEMP_DATA_DIRECTORY is not None:
        data_directory = pathlib.Path(TEMP_DATA_DIRECTORY)
    else:
        data_directory = pathlib.Path(__file__).parent.resolve() / "data"
    domain_folder = data_directory / strip_http_prefix(domain)
    if not domain_folder.exists():
        # Create the domain folder if it doesn't exist
        try:
            domain_folder.mkdir(parents=True, exist_ok=True)
            _logger.info(f"Created domain folder at {domain_folder}")
            return True
        except Exception as e:
            _logger.warning(f"Failed to create domain folder at {domain_folder}. Error: {e}")
            return False
    return True

def make_data_dir():
    global TEMP_DATA_DIRECTORY
    our_folder = pathlib.Path(__file__).parent.resolve()
    if not os.path.exists(our_folder / "data"):
        try:
            os.mkdir(our_folder / "data")
            _logger.info(f"Created data directory at {our_folder / 'data'}")
        except Exception as e:
            _logger.warning(f"Failed to create data directory at {our_folder / 'data'}. Using temp directory instead. Error: {e}")
            TEMP_DATA_DIRECTORY = tempfile.mkdtemp()
            _logger.info(f"Using temp data directory at {TEMP_DATA_DIRECTORY}")
    
    # Try to write into the data directory to make sure we have permissions
    test_file_path = (our_folder / "data" / "test.txt") if TEMP_DATA_DIRECTORY is None else (pathlib.Path(TEMP_DATA_DIRECTORY) / "test.txt")
    try:
        with open(test_file_path, 'w') as f:
            f.write("test")
        with open(test_file_path, 'r') as f:
            if f.read() != "test":
                raise Exception("Failed to read back the test data from the data directory.")
        _logger.info(f"Successfully wrote to data directory at {test_file_path}")
    except Exception as e:
        _logger.error(f"Failed to write to data directory at {test_file_path}. Falling back to temp directory. Error: {e}")
        TEMP_DATA_DIRECTORY = tempfile.mkdtemp()
        _logger.info(f"Using temp data directory at {TEMP_DATA_DIRECTORY}")

def create_cookie(username, secret_key=DEFAULT_SECRET_KEY, is_admin: bool=True) -> str:
    data = {
        "is_admin": is_admin,
        "username": username
    }
    cookie = sign(data, secret_key)
    return cookie

_serializer_instance = None
def get_saved_cookie(domain="localhost:5001") -> Union[str, None]:
    global _serializer_instance
    global TEMP_DATA_DIRECTORY
    # The structure will be a folder with the domain name, and inside it will have credentials.txt which will have a registered user
    # It will also have a cookie.txt which will have a cookie for that user if authentication was successful
    if TEMP_DATA_DIRECTORY is not None:
        data_directory = pathlib.Path(TEMP_DATA_DIRECTORY)
    else:
        data_directory = pathlib.Path(__file__).parent.resolve() / "data"
    cookie_file_path = data_directory / strip_http_prefix(domain) / "cookie.txt"
    check_domain_folder_exists(domain)
    if cookie_file_path.exists():
        try:
            with open(cookie_file_path, 'r') as f:
                cookie = f.read().strip()
        except Exception as e:
            _logger.warning(f"Failed to read cookie from {cookie_file_path}. Error: {e}")
            return None
        
        if _serializer_instance is None:
            # If there is no serializer create it.
            _serializer_instance = flask_unsign.session.get_serializer(secret_key=DEFAULT_SECRET_KEY, salt=DEFAULT_SALT, legacy=False)

        # We don't actually care about the secret key since we aren't re-signing the cookie
        # We just want the data, and the timestamp to check if the cookie is still valid based on the timestamp.
        is_valid, timestamp = _serializer_instance.loads_unsafe(cookie)
        valid_cookie = True
        if timestamp is None:
            _logger.debug("[-] Failed to parse timestamp from cookie. Cookie is likely invalid.")
            valid_cookie = False
        # Check if the cookie is expired based on the timestamp and the max time allowed.
        elif timestamp + MAX_TIME_ALLOWED < datetime.datetime.now().timestamp():
            _logger.info(f"[-] Cookie from {cookie_file_path} is expired based on timestamp. Timestamp: {timestamp}")
            valid_cookie = False
        
        if valid_cookie:
            _logger.debug(f"[+] Successfully loaded cookie from {cookie_file_path} and it is still valid based on timestamp.")
            return cookie
    
        _logger.debug(f"[-] Cookie from {cookie_file_path} is not valid. Deleting cookie file and re-authenticating.")
        try:
            cookie_file_path.unlink()
        except Exception as e:
            _logger.warning(f"Failed to delete invalid cookie file at {cookie_file_path}. Error: {e}")
        
    # At this point either the cookie file doesn't exist, or it is invalid.
    # Get the credentials file to see if we have a registered user we can authenticate with to get a new cookie.
    credentials_file_path = data_directory / "data" / strip_http_prefix(domain) / "credentials.txt"
    username = None
    password = None
    if credentials_file_path.exists():
        try:
            with open(credentials_file_path, 'r') as f:
                # The credentials file will have the username on the first line, and the password on the second line.
                username = f.readline().strip()
                password = f.readline().strip()
        except Exception as e:
            _logger.warning(f"Failed to read credentials from {credentials_file_path}. Using default credentials. Error: {e}")

    if username is None or password is None:
        _logger.warning(f"Failed to get valid credentials from {credentials_file_path}. Using default credentials.")
        username = 'green_testing_user'
        password = '8&bN4z@Cj2WqP'

    if 'mgmt' in domain:
        # If the domain is for the backend, then we need to use the backend's auth endpoint to get a cookie.
        pass
    elif 'admin' in domain:
        # If the domain is for the admin, then we need to use the admin's auth endpoint to get a cookie.
        pass
    elif 'www' in domain:
        # If the domain is for the frontend, then we need to use the frontend's auth endpoint to get a cookie.
        pass

def save_flag(domain: str, flag: str, type: str):
    global TEMP_DATA_DIRECTORY
    if TEMP_DATA_DIRECTORY is not None:
        data_directory = pathlib.Path(TEMP_DATA_DIRECTORY)
    else:
        data_directory = pathlib.Path(__file__).parent.resolve() / "data"
    flag_file_path = data_directory / strip_http_prefix(domain) / "flags.txt"
    # Check that the domain folder exists, and if not create it.
    check_domain_folder_exists(domain)
    # Save the flag to the flags.txt file in the domain folder. It should be the format of "domain (type): flag"
    try:
        with open(flag_file_path, 'a') as f:
            f.write(f"{domain} ({type}): {flag}\n")
        _logger.info(f"Successfully saved flag \"{flag}\" for {domain} to {flag_file_path}")
    except Exception as e:
        _logger.warning(f"Failed to save flag \"{flag}\" for {domain} to {flag_file_path}. Error: {e}")
        